/////////////////////////////////////////////////////////////
// TpiNetClient.cs
//
// This file implements a Form-derived class that is also
// a remote TPI.NET client. It demonstrates how to connect
// to a remote TLA and subscribe to events raised by TPI.NET.
/////////////////////////////////////////////////////////////

using System;
using System.Drawing;
using System.Collections;
using System.ComponentModel;
using System.Windows.Forms;
using System.Data;
using Tektronix.LogicAnalyzer.TpiNet;

namespace CSharpRemoteTpiNetApplication
{
	public class CSRemoteClient : System.Windows.Forms.Form
	{
        // Data members for controls
		private System.Windows.Forms.Label versionLabel;
        private System.Windows.Forms.Button runButton;
        private System.Windows.Forms.Label label1;
        private System.Windows.Forms.Label eventLabel;

        /// <summary>
		/// Required designer variable.
		/// </summary>
		private System.ComponentModel.Container components = null;

		// A reference to the top of the TPI.NET object hierarchy
		// in the remote TLA application
		private ITlaSystem m_system = null;

        // Keep track of how many acquisitions we have run.
        private int m_startCount = 0;
        private int m_endCount = 0;

		// Delegates that are required to subscribe to
		// events raised by the remote TLA application.
		private EventHandler m_runStartHandler;
        private EventHandler m_runCompleteHandler;


		// The constructor connects to the remote TLA application and 
		// subscribes to two of the ITlaRunControl events.
		public CSRemoteClient()
		{
			// Required for Windows Form Designer support
			InitializeComponent();
			
			string version = null;
			
			// The assembly TlaNetInterfaces, which this project references,
			// contains a utility class named RemoteTlaSystem. It has a
			// Connect() static method which is used to set up a connection
			// to a remote TLA. To connect, an application configuration file
            // named ITlaSystemRemoteClient.config is used. This file contains
			// the information needed to configure this application to
			// communicate with a TLA running on the local host. The
			// configuration file can be modified to connect to a TLA
			// on a network.
			// NOTE: For the call to Connect() to succeed, the file
            // ITlaSystemRemoteClient.config must be in the same directory as
			// the application executable.
			try 
			{
                m_system = RemoteTlaSystem.Connect("ITlaSystemRemoteClient.config");
			    version = m_system.SWVersion;
			}
			catch 
            {
			    // Failed to connect.
				MessageBox.Show("Could not connect to the TLA application. Make sure the TLA application is running before starting the client application. Also make sure the configuration file is located in the executable directory.");
			    m_system = null;
			}

            if (null == m_system) 
            {
			    versionLabel.Text = "Could not connect to TLA application";
			    runButton.Enabled = false;
			    runButton.Text = "Cannot run TLA";
			}
			else 
            {
				// Connection was successful. Now subscribe to TPI.NET events.
                SubscribeToTpiNetEvents();
				
				versionLabel.Text = string.Format("TLA Software Version is {0}", version);
				
				// Local events don't need shims.
			    runButton.Click += new EventHandler(OnRunButton);
			}
		}

        // Add TPI.NET event subscriptions.
        private void SubscribeToTpiNetEvents()
        {
            if (m_system == null) return;

            // Subscribe to ITlaRunControl.RunStarted.
            m_runStartHandler = EventRemoter.Create(new EventHandler(OnRunStarted));
            m_system.RunControl.RunStarted += m_runStartHandler;

            // Subscribe to ITlaRunControl.RunCompleted.
            m_runCompleteHandler = EventRemoter.Create(new EventHandler(OnRunCompleted));
            m_system.RunControl.RunCompleted += m_runCompleteHandler;
        }

        // Removes TPI.NET event subscriptions when the user exits the
        // application. TPI.NET clients should always clean up their
        // subscriptions before exiting.
        private void UnsubscribeFromTpiNetEvents()
        {
            if (m_system == null) return;

            // Unsubscribe from the run events.
            m_system.RunControl.RunStarted -= m_runStartHandler;
            m_system.RunControl.RunCompleted -= m_runCompleteHandler;

            m_runStartHandler = null;
            m_runCompleteHandler = null;
        }

        private bool _disposed = false;

        /// <summary>
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose(bool disposing)
		{
            if (!_disposed)
            {
                try
                {
                    if (disposing)
                    {
                        UnsubscribeFromTpiNetEvents();

				        if (components != null) 
				        {
					        components.Dispose();
				        }
                        components = null;
                    }
                    base.Dispose(disposing);
                }
                catch
                {
                    _disposed = false;
                    throw;
                }
                _disposed = true;
            }
        }

		#region Windows Form Designer generated code
		/// <summary>
		/// Required method for Designer support - do not modify
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
            this.versionLabel = new System.Windows.Forms.Label();
            this.runButton = new System.Windows.Forms.Button();
            this.label1 = new System.Windows.Forms.Label();
            this.eventLabel = new System.Windows.Forms.Label();
            this.SuspendLayout();
            // 
            // versionLabel
            // 
            this.versionLabel.Location = new System.Drawing.Point(16, 24);
            this.versionLabel.Name = "versionLabel";
            this.versionLabel.Size = new System.Drawing.Size(256, 32);
            this.versionLabel.TabIndex = 0;
            this.versionLabel.Text = "label1";
            // 
            // runButton
            // 
            this.runButton.Location = new System.Drawing.Point(16, 64);
            this.runButton.Name = "runButton";
            this.runButton.Size = new System.Drawing.Size(248, 32);
            this.runButton.TabIndex = 1;
            this.runButton.Text = "Run the TLA";
            // 
            // label1
            // 
            this.label1.Location = new System.Drawing.Point(16, 120);
            this.label1.Name = "label1";
            this.label1.Size = new System.Drawing.Size(256, 24);
            this.label1.TabIndex = 2;
            this.label1.Text = "Most Recent Run Event:";
            // 
            // eventLabel
            // 
            this.eventLabel.Location = new System.Drawing.Point(16, 152);
            this.eventLabel.Name = "eventLabel";
            this.eventLabel.Size = new System.Drawing.Size(256, 32);
            this.eventLabel.TabIndex = 3;
            this.eventLabel.Text = "None.";
            // 
            // CSRemoteClient
            // 
            this.AutoScaleBaseSize = new System.Drawing.Size(6, 15);
            this.ClientSize = new System.Drawing.Size(292, 208);
            this.Controls.AddRange(new System.Windows.Forms.Control[] {
                                                                          this.eventLabel,
                                                                          this.label1,
                                                                          this.runButton,
                                                                          this.versionLabel});
            this.Name = "CSRemoteClient";
            this.Text = "Remote TPI.NET Client";
            this.ResumeLayout(false);

        }
		#endregion

		/// <summary>
		/// The main entry point for the application.
		/// </summary>
		[STAThread]
		static void Main() 
		{
			try 
            {
			    Application.Run(new CSRemoteClient());
			}
			catch
            {
			    MessageBox.Show("The remote TPI.NET client experienced an error and must shut down.");
			}
		}
		
	// Event Handlers	

        // Event handler for the ITlaRunControl.RunStarted event.
        private void OnRunStarted(object sender, EventArgs args)
        {
            // Asynchronous events from the TLA Application come in on worker threads.
            // In order to update the GUI controls, we must be on the UI thread.
            if (InvokeRequired)
            {
                // Make a call to the UI thread to output the results.
                BeginInvoke(new EventHandler(OnRunStarted), new object[] {sender, args});
            }
            else
            {
                // We're already on the UI thread.  Output the results directly.
                m_startCount++;
                eventLabel.Text = String.Format("Acquisition {0} started.", m_startCount);
                runButton.Enabled = false;
            }
        }

        // Event handler for the ITlaRunControl.RunCompleted event.
        private void OnRunCompleted(object sender, EventArgs args)
        {
            // Asynchronous events from the TLA Application come in on worker threads.
            // In order to update the GUI controls, we must be on the UI thread.
            if (InvokeRequired)
            {
                // Make a call to the UI thread to output the results.
                BeginInvoke(new EventHandler(OnRunCompleted), new object[] {sender, args});
            }
            else
            {
                // We're already on the UI thread.  Output the results directly.
                m_endCount++;
                eventLabel.Text = String.Format("Acquisition {0} completed.", m_endCount);
                runButton.Enabled = true;
            }
        }
		
        // Starts a TLA acquistion when the user clicks the "Run the TLA" button.
		private void OnRunButton(object sender, EventArgs args) 
		{
            if (m_system == null) return;

            m_system.RunControl.Run();
		}
	}
}
